/*
 * ESP32 Hardware Abstraction Layer - Peripheral Manager
 * 
 * SPDX-FileCopyrightText: 2019-2023 Espressif Systems (Shanghai) CO LTD
 * SPDX-License-Identifier: Apache-2.0
 */

#ifndef HAL_ESP32_HAL_PERIMAN_H_
#define HAL_ESP32_HAL_PERIMAN_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>
#include "soc/soc_caps.h"

/**
 * @brief Peripheral bus type enumeration
 * 
 * Defines all supported peripheral bus types for GPIO pin assignment.
 * Each peripheral type represents a different function that can be
 * assigned to a GPIO pin through the peripheral manager.
 */
typedef enum {
    ESP32_BUS_TYPE_INIT = 0,        ///< Initial/unassigned state
    ESP32_BUS_TYPE_GPIO,            ///< General Purpose I/O
    ESP32_BUS_TYPE_UART_RX,         ///< UART Receive pin
    ESP32_BUS_TYPE_UART_TX,         ///< UART Transmit pin
    ESP32_BUS_TYPE_UART_CTS,        ///< UART Clear to Send
    ESP32_BUS_TYPE_UART_RTS,        ///< UART Request to Send
#if SOC_I2C_SUPPORTED
    ESP32_BUS_TYPE_I2C_MASTER_SDA,  ///< I2C Data line (Master mode)
    ESP32_BUS_TYPE_I2C_MASTER_SCL,  ///< I2C Clock line (Master mode)
#endif
#if SOC_LEDC_SUPPORTED
    ESP32_BUS_TYPE_LEDC,            ///< LED Controller PWM
#endif
#if SOC_RMT_SUPPORTED
    ESP32_BUS_TYPE_RMT_TX,          ///< Remote Control Transmit
    ESP32_BUS_TYPE_RMT_RX,          ///< Remote Control Receive
#endif
#if SOC_SDM_SUPPORTED
    ESP32_BUS_TYPE_SIGMADELTA,      ///< Sigma-Delta Modulator
#endif
#if SOC_ADC_SUPPORTED
    ESP32_BUS_TYPE_ADC_ONESHOT,     ///< ADC One-shot reading mode
    ESP32_BUS_TYPE_ADC_CONTINUOUS,  ///< ADC Continuous reading mode
#endif
#if SOC_TOUCH_SENSOR_SUPPORTED
    ESP32_BUS_TYPE_TOUCH,           ///< Touch sensor
#endif
#if SOC_DAC_SUPPORTED
    ESP32_BUS_TYPE_DAC_ONESHOT,     ///< DAC One-shot output
    ESP32_BUS_TYPE_DAC_CONTINUOUS,  ///< DAC Continuous output
    ESP32_BUS_TYPE_DAC_COSINE,      ///< DAC Cosine wave output
#endif
#if SOC_SPI_SUPPORTED
    ESP32_BUS_TYPE_SPI_MASTER_SCK,  ///< SPI Master Clock
    ESP32_BUS_TYPE_SPI_MASTER_MISO, ///< SPI Master Input
    ESP32_BUS_TYPE_SPI_MASTER_MOSI, ///< SPI Master Output
    ESP32_BUS_TYPE_SPI_MASTER_SS,   ///< SPI Master Slave Select
    ESP32_BUS_TYPE_SPI_SLAVE_SCK,   ///< SPI Slave Clock
    ESP32_BUS_TYPE_SPI_SLAVE_MISO,  ///< SPI Slave Input
    ESP32_BUS_TYPE_SPI_SLAVE_MOSI,  ///< SPI Slave Output
    ESP32_BUS_TYPE_SPI_SLAVE_SS,    ///< SPI Slave Slave Select
#endif
#if SOC_I2S_SUPPORTED
    ESP32_BUS_TYPE_I2S_STD_MCLK,    ///< I2S Master Clock
    ESP32_BUS_TYPE_I2S_STD_BCLK,    ///< I2S Bit Clock
    ESP32_BUS_TYPE_I2S_STD_WS,      ///< I2S Word Select
    ESP32_BUS_TYPE_I2S_STD_DOUT,    ///< I2S Data Output
    ESP32_BUS_TYPE_I2S_STD_DIN,     ///< I2S Data Input
#endif
#if SOC_PCNT_SUPPORTED
    ESP32_BUS_TYPE_PCNT_SIG,        ///< Pulse Counter Signal
    ESP32_BUS_TYPE_PCNT_CTRL,       ///< Pulse Counter Control
#endif
#if SOC_MCPWM_SUPPORTED
    ESP32_BUS_TYPE_MCPWM,           ///< Motor Control PWM
#endif
#if SOC_SDMMC_HOST_SUPPORTED
    ESP32_BUS_TYPE_SDMMC_CLK,       ///< SD/MMC Clock
    ESP32_BUS_TYPE_SDMMC_CMD,       ///< SD/MMC Command
    ESP32_BUS_TYPE_SDMMC_D0,        ///< SD/MMC Data 0
    ESP32_BUS_TYPE_SDMMC_D1,        ///< SD/MMC Data 1
    ESP32_BUS_TYPE_SDMMC_D2,        ///< SD/MMC Data 2
    ESP32_BUS_TYPE_SDMMC_D3,        ///< SD/MMC Data 3
#endif
#if SOC_USB_OTG_SUPPORTED
    ESP32_BUS_TYPE_USB_OTG_DP,      ///< USB OTG Data Plus
    ESP32_BUS_TYPE_USB_OTG_DM,      ///< USB OTG Data Minus
#endif
    ESP32_BUS_TYPE_MAX              ///< Maximum value (boundary marker)
} peripheral_bus_type_t;

/**
 * @brief Peripheral bus deinitialization callback function type
 * 
 * This callback is invoked when a pin is being released from its current
 * peripheral assignment. It allows peripherals to perform proper cleanup
 * of their resources before the pin is reassigned.
 * 
 * @param bus Pointer to the bus handle/context to deinitialize
 * @return true if deinitialization was successful, false otherwise
 */
typedef bool (*peripheral_bus_deinit_cb_t)(void *bus);

/**
 * @brief Pin validation macro
 * 
 * Checks if a pin number is valid for GPIO operations on the current SoC.
 * Some pins may be reserved for internal use or not available on all variants.
 * 
 * @param p Pin number to validate
 * @return true if pin is invalid, false if valid
 */
#define GPIO_NOT_VALID(p) ((p >= SOC_GPIO_PIN_COUNT) || ((SOC_GPIO_VALID_GPIO_MASK & (1ULL << p)) == 0))

/**
 * @brief Assign a GPIO pin to a specific peripheral bus
 * 
 * This function registers a GPIO pin with the peripheral manager and assigns
 * it to the specified peripheral type. If the pin is already assigned to
 * another peripheral, the assignment will fail unless the pin is cleared first.
 * 
 * @param pin GPIO pin number (0 to SOC_GPIO_PIN_COUNT-1)
 * @param type Peripheral bus type from peripheral_bus_type_t enum
 * @param bus Pointer to peripheral bus handle/context (can be NULL)
 * @param bus_num Bus number identifier (-1 if not applicable)
 * @param bus_channel Channel number within the bus (-1 if not applicable)
 * 
 * @return true if pin was successfully assigned, false on error
 * 
 * @note Always check return value for error handling
 * @note Pin must be valid for GPIO operations (check with GPIO_NOT_VALID macro)
 * @note If pin is already assigned, use perimanClearPinBus() first
 */
bool perimanSetPinBus(uint8_t pin, peripheral_bus_type_t type, 
                      void *bus, int8_t bus_num, int8_t bus_channel);

/**
 * @brief Release a GPIO pin from its current peripheral assignment
 * 
 * This function releases a pin from its current peripheral assignment and
 * resets it to the unassigned state. If a deinitialization callback is
 * registered for the peripheral type, it will be called to perform cleanup.
 * 
 * @param pin GPIO pin number to release
 * 
 * @return true if pin was successfully released, false on error
 * 
 * @note Safe to call on unassigned pins
 * @note Will invoke registered deinit callback if available
 * @note Pin state is reset to ESP32_BUS_TYPE_INIT after successful release
 */
bool perimanClearPinBus(uint8_t pin);

/**
 * @brief Get the bus handle for a pin assigned to a specific peripheral type
 * 
 * Retrieves the bus handle that was stored when the pin was assigned to
 * the specified peripheral type. This allows peripherals to retrieve their
 * context information for pin operations.
 * 
 * @param pin GPIO pin number to query
 * @param type Expected peripheral bus type
 * 
 * @return Bus handle if pin is assigned to the specified type, NULL otherwise
 * 
 * @note Returns NULL if pin is not assigned to the specified peripheral type
 * @note Use perimanGetPinBusType() first to check current assignment
 */
void *perimanGetPinBus(uint8_t pin, peripheral_bus_type_t type);

/**
 * @brief Get the current peripheral bus type assignment for a GPIO pin
 * 
 * Returns the current peripheral type assigned to the specified pin.
 * Can be used to check if a pin is available or to determine its current use.
 * 
 * @param pin GPIO pin number to query
 * 
 * @return Current peripheral bus type assigned to the pin
 * @retval ESP32_BUS_TYPE_INIT if pin is not assigned to any peripheral
 * 
 * @note Always returns a valid peripheral_bus_type_t value
 * @note ESP32_BUS_TYPE_INIT indicates the pin is available for assignment
 */
peripheral_bus_type_t perimanGetPinBusType(uint8_t pin);

/**
 * @brief Get human-readable string name for a peripheral bus type
 * 
 * Returns a constant string representing the name of the specified
 * peripheral bus type. Useful for logging and debugging purposes.
 * 
 * @param type Peripheral bus type enum value
 * 
 * @return Constant string representing the bus type name
 * 
 * @note Never returns NULL (returns "UNKNOWN" for invalid types)
 * @note String is statically allocated and should not be freed
 * @note Useful for debug output and error messages
 */
const char *perimanGetTypeName(peripheral_bus_type_t type);

/**
 * @brief Register a deinitialization callback for a peripheral bus type
 * 
 * Registers a callback function that will be invoked when any pin assigned
 * to the specified peripheral type is being released. This allows peripherals
 * to perform proper cleanup of their resources.
 * 
 * @param type Peripheral bus type to register callback for
 * @param cb Callback function pointer (NULL to unregister)
 * 
 * @note Only one callback per bus type is supported
 * @note Passing NULL removes the callback
 * @note Callback is invoked during perimanClearPinBus() operations
 * @note Should be called during peripheral initialization
 */
void perimanSetBusDeinit(peripheral_bus_type_t type, peripheral_bus_deinit_cb_t cb);

/**
 * @brief Check if a pin is currently assigned to any peripheral
 * 
 * Convenience function to check if a pin is available for assignment.
 * 
 * @param pin GPIO pin number to check
 * @return true if pin is available (not assigned), false if in use
 */
static inline bool perimanPinIsAvailable(uint8_t pin) {
    return perimanGetPinBusType(pin) == ESP32_BUS_TYPE_INIT;
}

/**
 * @brief Validate pin number for GPIO operations
 * 
 * Checks if the specified pin number is valid for GPIO operations
 * on the current ESP32 variant.
 * 
 * @param pin Pin number to validate
 * @return true if pin is valid, false if invalid
 */
static inline bool perimanPinIsValid(uint8_t pin) {
    return !GPIO_NOT_VALID(pin);
}

/**
 * @brief Force clear a pin without calling deinit callbacks
 * 
 * Internal function to force clear a pin assignment without invoking
 * deinitialization callbacks. Use with caution as it may cause resource leaks.
 * 
 * @param pin GPIO pin number to force clear
 * @return true if cleared successfully, false on error
 * 
 * @warning This function bypasses normal cleanup procedures
 * @note Primarily for internal use and emergency cleanup scenarios
 */
bool perimanForceClearPinBus(uint8_t pin);

#ifdef __cplusplus
}
#endif

#endif /* HAL_ESP32_HAL_PERIMAN_H_ */