/*
 *  Copyright (c) 2020 Temporal Technologies, Inc. All Rights Reserved
 *
 *  Copyright 2012-2016 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Modifications copyright (C) 2017 Uber Technologies, Inc.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not
 *  use this file except in compliance with the License. A copy of the License is
 *  located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 *  or in the "license" file accompanying this file. This file is distributed on
 *  an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *  express or implied. See the License for the specific language governing
 *  permissions and limitations under the License.
 */

package io.temporal.samples.hello;

import io.temporal.activity.ActivityInterface;
import io.temporal.activity.ActivityMethod;
import io.temporal.activity.LocalActivityOptions;
import io.temporal.client.WorkflowClient;
import io.temporal.client.WorkflowOptions;
import io.temporal.common.RetryOptions;
import io.temporal.failure.ApplicationFailure;
import io.temporal.serviceclient.WorkflowServiceStubs;
import io.temporal.worker.Worker;
import io.temporal.worker.WorkerFactory;
import io.temporal.workflow.Workflow;
import io.temporal.workflow.WorkflowInterface;
import io.temporal.workflow.WorkflowMethod;
import java.time.Duration;

/**
 * Hello World Temporal workflow that executes a single local activity. Requires a local instance
 * the Temporal service to be running.
 *
 * <p>Some of the Activities are very short lived and do not need the queuing semantic, flow
 * control, rate limiting and routing capabilities. For these Temporal supports so called local
 * Activity feature. Local Activities are executed in the same worker process as the Workflow that
 * invoked them. Consider using local Activities for functions that are:
 *
 * <ul>
 *   <li>no longer than a few seconds
 *   <li>do not require global rate limiting
 *   <li>do not require routing to specific workers or pools of workers
 *   <li>can be implemented in the same binary as the Workflow that invokes them
 * </ul>
 *
 * <p>The main benefit of local Activities is that they are much more efficient in utilizing
 * Temporal service resources and have much lower latency overhead comparing to the usual Activity
 * invocation.
 */
public class HelloLocalActivity {

  static final String TASK_QUEUE = "HelloLocalActivity";

  public static void main(String[] args) {
    // gRPC stubs wrapper that talks to the local docker instance of temporal service.
    WorkflowServiceStubs service = WorkflowServiceStubs.newLocalServiceStubs();
    // client that can be used to start and signal workflows
    WorkflowClient client = WorkflowClient.newInstance(service);

    // worker factory that can be used to create workers for specific task queues
    WorkerFactory factory = WorkerFactory.newInstance(client);
    // Worker that listens on a task queue and hosts both workflow and activity implementations.
    Worker worker = factory.newWorker(TASK_QUEUE);
    // Workflows are stateful. So you need a type to create instances.
    worker.registerWorkflowImplementationTypes(GreetingWorkflowImpl.class);
    // Activities are stateless and thread safe. So a shared instance is used.
    worker.registerActivitiesImplementations(new GreetingLocalActivityImpl());
    // Start listening to the workflow and activity task queues.
    factory.start();

    // Start a workflow execution. Usually this is done from another program.
    // Uses task queue from the GreetingWorkflow @WorkflowMethod annotation.
    GreetingWorkflow workflow =
        client.newWorkflowStub(
            GreetingWorkflow.class, WorkflowOptions.newBuilder().setTaskQueue(TASK_QUEUE).build());
    // Execute a workflow waiting for it to complete. See {@link
    // io.temporal.samples.hello.HelloSignal}
    // for an example of starting workflow without waiting synchronously for its result.
    String greeting = workflow.getGreeting("World");
    System.out.println(greeting);
    System.exit(0);
  }

  /**
   * The Workflow Definition's Interface must contain one method annotated with @WorkflowMethod.
   *
   * <p>Workflow Definitions should not contain any heavyweight computations, non-deterministic
   * code, network calls, database operations, etc. Those things should be handled by the
   * Activities.
   *
   * @see io.temporal.workflow.WorkflowInterface
   * @see io.temporal.workflow.WorkflowMethod
   */
  @WorkflowInterface
  public interface GreetingWorkflow {
    @WorkflowMethod
    String getGreeting(String name);
  }

  /** Activity interface is just a POJO. */
  @ActivityInterface
  public interface GreetingActivities {
    @ActivityMethod
    String composeGreeting(String greeting, String name);
  }

  /** GreetingWorkflow implementation that calls GreetingsActivities#composeGreeting. */
  public static class GreetingWorkflowImpl implements GreetingWorkflow {
    private final GreetingActivities activities;

    {
      RetryOptions build =
          RetryOptions.newBuilder()
              // .setMaximumInterval(Duration.ofSeconds(2))
              .setMaximumAttempts(0)
              // .setMaximumAttempts(2)
              .build();
      activities =
          Workflow.newLocalActivityStub(
              GreetingActivities.class,
              LocalActivityOptions.newBuilder()
                  .setRetryOptions(build)
                  .setStartToCloseTimeout(Duration.ofSeconds(2))
                  .build());
    }

    @Override
    public String getGreeting(String name) {
      // This is a blocking call that returns only after the activity has completed.
      return activities.composeGreeting("Hello", name);
    }
  }

  static class GreetingLocalActivityImpl implements GreetingActivities {
    @Override
    public String composeGreeting(String greeting, String name) {

      System.out.println(">>>>> retrying ");

      throw ApplicationFailure.newFailure("should retry", "MyError");

      //      try {
      //        Thread.sleep(11000);
      //      } catch (InterruptedException e) {
      //        throw new RuntimeException(e);
      //      }
      //
      //      return greeting + " " + name + "!";
    }
  }
}
