/*
 * Decompiled with CFR 0.152.
 */
package oracle.nosql.driver.util;

import java.sql.Timestamp;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import oracle.nosql.driver.util.CheckNull;

public class TimestampUtil {
    private static final String DEFAULT_PATTERN = "uuuu-MM-dd['T'HH:mm:ss]";
    private static final ZoneId UTCZone = ZoneId.of(ZoneOffset.UTC.getId());
    private static final int MAX_NUMBER_FRACSEC = 9;
    private static final char[] compSeparators = new char[]{'-', '-', 'T', ':', ':', '.'};
    private static final String[] compNames = new String[]{"year", "month", "day", "hour", "minute", "second", "fractional second"};

    public static Timestamp parseString(String text) {
        return TimestampUtil.parseString(text, null, true, true);
    }

    public static Timestamp parseString(String text, String pattern, boolean withZoneUTC, boolean optionalFracSecond) {
        CheckNull.requireNonNull(text, "Timestamp string must be non-null");
        boolean optionalZoneOffset = false;
        if (pattern == null || pattern.equals(DEFAULT_PATTERN)) {
            String tsStr = TimestampUtil.trimUTCZoneOffset(text);
            if (tsStr != null) {
                return TimestampUtil.parseWithDefaultPattern(tsStr);
            }
            optionalZoneOffset = true;
        }
        String fmt = pattern != null ? pattern : DEFAULT_PATTERN;
        try {
            boolean hasOffset;
            DateTimeFormatter dtf = TimestampUtil.getDateTimeFormatter(fmt, withZoneUTC, optionalFracSecond, optionalZoneOffset);
            TemporalAccessor ta = dtf.parse(text);
            if (!(ta.isSupported(ChronoField.YEAR) && ta.isSupported(ChronoField.MONTH_OF_YEAR) && ta.isSupported(ChronoField.DAY_OF_MONTH))) {
                throw new IllegalArgumentException("The timestamp string must contain year, month and day");
            }
            boolean bl = hasOffset = ta.isSupported(ChronoField.OFFSET_SECONDS) && ta.get(ChronoField.OFFSET_SECONDS) != 0;
            Instant instant = ta.isSupported(ChronoField.HOUR_OF_DAY) ? (hasOffset ? OffsetDateTime.from(ta).toInstant() : Instant.from(ta)) : LocalDate.from(ta).atStartOfDay(hasOffset ? ZoneOffset.from(ta) : UTCZone).toInstant();
            return TimestampUtil.toTimestamp(instant);
        }
        catch (IllegalArgumentException iae) {
            throw new IllegalArgumentException("Failed to parse the date string '" + text + "' with the pattern: " + fmt + ": " + iae.getMessage(), iae);
        }
        catch (DateTimeParseException dtpe) {
            throw new IllegalArgumentException("Failed to parse the date string '" + text + "' with the pattern: " + fmt + ": " + dtpe.getMessage(), dtpe);
        }
        catch (DateTimeException dte) {
            throw new IllegalArgumentException("Failed to parse the date string '" + text + "' with the pattern: " + fmt + ": " + dte.getMessage(), dte);
        }
    }

    private static String trimUTCZoneOffset(String ts) {
        if (ts.endsWith("Z")) {
            return ts.substring(0, ts.length() - 1);
        }
        if (ts.endsWith("+00:00")) {
            return ts.substring(0, ts.length() - 6);
        }
        if (!TimestampUtil.hasSignOfZoneOffset(ts)) {
            return ts;
        }
        return null;
    }

    private static boolean hasSignOfZoneOffset(String ts) {
        if (ts.indexOf(43) > 0) {
            return true;
        }
        int pos = 0;
        for (int i = 0; i < 3; ++i) {
            if ((pos = ts.indexOf(45, pos + 1)) >= 0) continue;
            return false;
        }
        return true;
    }

    public static String formatString(Timestamp ts) {
        CheckNull.requireNonNull(ts, "Timestamp must be non-null");
        return TimestampUtil.formatString(ts, null, true, true);
    }

    private static String formatString(Timestamp timestamp, String pattern, boolean withZoneUTC, boolean optionalFracSecond) {
        CheckNull.requireNonNull(timestamp, "Timestamp must be non-null");
        String fmt = pattern == null ? DEFAULT_PATTERN : pattern;
        try {
            ZonedDateTime zdt = TimestampUtil.toUTCDateTime(timestamp);
            return zdt.format(TimestampUtil.getDateTimeFormatter(fmt, withZoneUTC, optionalFracSecond, true));
        }
        catch (IllegalArgumentException iae) {
            throw new IllegalArgumentException("Failed to format the timestamp with pattern '" + fmt + "': " + iae.getMessage(), iae);
        }
        catch (DateTimeException dte) {
            throw new IllegalArgumentException("Failed to format the timestamp with pattern '" + fmt + "': " + dte.getMessage(), dte);
        }
    }

    private static Timestamp parseWithDefaultPattern(String ts) {
        int i;
        int[] comps = new int[7];
        int comp = 0;
        int val = 0;
        int ndigits = 0;
        int len = ts.length();
        boolean isBC = ts.charAt(0) == '-';
        int n = i = isBC ? 1 : 0;
        while (i < len) {
            char ch = ts.charAt(i);
            if (comp < 6) {
                switch (ch) {
                    case '0': 
                    case '1': 
                    case '2': 
                    case '3': 
                    case '4': 
                    case '5': 
                    case '6': 
                    case '7': 
                    case '8': 
                    case '9': {
                        val = val * 10 + (ch - 48);
                        ++ndigits;
                        break;
                    }
                    default: {
                        if (ch == compSeparators[comp]) {
                            TimestampUtil.checkAndSetValue(comps, comp, val, ndigits, ts);
                            ++comp;
                            val = 0;
                            ndigits = 0;
                            break;
                        }
                        TimestampUtil.raiseParseError(ts, "invalid character '" + ch + "' while parsing component " + compNames[comp]);
                        break;
                    }
                }
            } else {
                if (comp != 6) {
                    throw new IllegalArgumentException("Invalid component index: " + comp + ", it is expected to be 6");
                }
                switch (ch) {
                    case '0': 
                    case '1': 
                    case '2': 
                    case '3': 
                    case '4': 
                    case '5': 
                    case '6': 
                    case '7': 
                    case '8': 
                    case '9': {
                        val = val * 10 + (ch - 48);
                        ++ndigits;
                        break;
                    }
                    default: {
                        TimestampUtil.raiseParseError(ts, "invalid character '" + ch + "' while parsing component " + compNames[comp]);
                    }
                }
            }
            ++i;
        }
        TimestampUtil.checkAndSetValue(comps, comp, val, ndigits, ts);
        if (comp < 2) {
            TimestampUtil.raiseParseError(ts, "the timestamp string must have at least the 3 date components");
        }
        if (comp == 6 && comps[6] > 0) {
            if (ndigits > 9) {
                TimestampUtil.raiseParseError(ts, "the fractional-seconds part contains more than 9 digits");
            } else if (ndigits < 9) {
                comps[6] = comps[6] * (int)Math.pow(10.0, 9 - ndigits);
            }
        }
        if (isBC) {
            comps[0] = -comps[0];
        }
        return TimestampUtil.createTimestamp(comps);
    }

    private static void checkAndSetValue(int[] comps, int comp, int value, int ndigits, String ts) {
        if (ndigits == 0) {
            TimestampUtil.raiseParseError(ts, "component " + compNames[comp] + "has 0 digits");
        }
        comps[comp] = value;
    }

    private static void raiseParseError(String ts, String err) {
        String errMsg = "Failed to parse the timestamp string '" + ts + "' with the pattern: " + DEFAULT_PATTERN + ": ";
        throw new IllegalArgumentException(errMsg + err);
    }

    private static void validateComponent(int index, int value) {
        switch (index) {
            case 1: {
                if (value >= 1 && value <= 12) break;
                throw new IllegalArgumentException("Invalid month, it should be in range from 1 to 12: " + value);
            }
            case 2: {
                if (value >= 1 && value <= 31) break;
                throw new IllegalArgumentException("Invalid day, it should be in range from 1 to 31: " + value);
            }
            case 3: {
                if (value >= 0 && value <= 23) break;
                throw new IllegalArgumentException("Invalid hour, it should be in range from 0 to 23: " + value);
            }
            case 4: {
                if (value >= 0 && value <= 59) break;
                throw new IllegalArgumentException("Invalid minute, it should be in range from 0 to 59: " + value);
            }
            case 5: {
                if (value >= 0 && value <= 59) break;
                throw new IllegalArgumentException("Invalid second, it should be in range from 0 to 59: " + value);
            }
            case 6: {
                if (value >= 0 && value <= 999999999) break;
                throw new IllegalArgumentException("Invalid second, it should be in range from 0 to 999999999: " + value);
            }
        }
    }

    private static Timestamp toTimestamp(Instant instant) {
        return TimestampUtil.createTimestamp(instant.getEpochSecond(), instant.getNano());
    }

    public static Timestamp createTimestamp(long seconds, int nanoSeconds) {
        Timestamp ts = new Timestamp(seconds * 1000L);
        ts.setNanos(nanoSeconds);
        return ts;
    }

    private static Timestamp createTimestamp(int[] comps) {
        if (comps.length < 3) {
            throw new IllegalArgumentException("Invalid timestamp components, it should contain at least 3 components: year, month and day, but only " + comps.length);
        }
        if (comps.length > 7) {
            throw new IllegalArgumentException("Invalid timestamp components, it should contain at most 7 components: year, month, day, hour, minute, second and nanosecond, but has " + comps.length + " components");
        }
        int num = comps.length;
        for (int i = 0; i < num; ++i) {
            TimestampUtil.validateComponent(i, comps[i]);
        }
        try {
            ZonedDateTime zdt = ZonedDateTime.of(comps[0], comps[1], comps[2], num > 3 ? comps[3] : 0, num > 4 ? comps[4] : 0, num > 5 ? comps[5] : 0, num > 6 ? comps[6] : 0, UTCZone);
            return TimestampUtil.toTimestamp(zdt.toInstant());
        }
        catch (DateTimeException dte) {
            throw new IllegalArgumentException("Invalid timestamp components: " + dte.getMessage());
        }
    }

    private static ZonedDateTime toUTCDateTime(Timestamp timestamp) {
        return TimestampUtil.toInstant(timestamp).atZone(UTCZone);
    }

    private static Instant toInstant(Timestamp timestamp) {
        return Instant.ofEpochSecond(TimestampUtil.getSeconds(timestamp), TimestampUtil.getNanoSeconds(timestamp));
    }

    public static long getSeconds(Timestamp timestamp) {
        long ms = timestamp.getTime();
        return ms > 0L ? ms / 1000L : (ms - 999L) / 1000L;
    }

    public static int getNanoSeconds(Timestamp timestamp) {
        return timestamp.getNanos();
    }

    private static DateTimeFormatter getDateTimeFormatter(String pattern, boolean withZoneUTC, boolean optionalFracSecond, boolean optionalOffset) {
        DateTimeFormatterBuilder dtfb = new DateTimeFormatterBuilder();
        dtfb.appendPattern(pattern);
        if (optionalFracSecond) {
            dtfb.optionalStart();
            dtfb.appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true);
            dtfb.optionalEnd();
        }
        if (optionalOffset) {
            dtfb.optionalStart();
            dtfb.appendOffset("+HH:MM", "Z");
            dtfb.optionalEnd();
        }
        return dtfb.toFormatter().withZone(withZoneUTC ? UTCZone : ZoneId.systemDefault());
    }
}

