/*
 * Decompiled with CFR 0.152.
 */
package oracle.nosql.driver.httpclient;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelInitializer;
import io.netty.channel.EventLoop;
import io.netty.channel.pool.ChannelPoolHandler;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.Promise;
import java.time.Instant;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import oracle.nosql.driver.util.LogUtil;

class ConnectionPool {
    static final int DEFAULT_INACTIVITY_PERIOD_SECS = 30;
    static final int DEFAULT_REFRESH_PERIOD_SECS = 30;
    static final int MAX_INACTIVITY_PERIOD_SECS = 30;
    private final Bootstrap bootstrap;
    private final ChannelPoolHandler handler;
    private final int poolMin;
    private final int inactivityPeriodSeconds;
    private final Logger logger;
    private KeepAlive keepAlive;
    private final ConcurrentLinkedDeque<Channel> queue;
    private final Map<Channel, ChannelStats> stats;
    private int acquiredChannelCount;

    ConnectionPool(Bootstrap bootstrap, final ChannelPoolHandler handler, Logger logger, boolean isMinimalPool, int poolMin, int inactivityPeriodSeconds) {
        this.bootstrap = bootstrap.clone();
        this.bootstrap.handler((ChannelHandler)new ChannelInitializer<Channel>(){

            protected void initChannel(Channel ch) throws Exception {
                assert (ch.eventLoop().inEventLoop());
                handler.channelCreated(ch);
            }
        });
        this.handler = handler;
        this.logger = logger;
        this.poolMin = poolMin;
        this.inactivityPeriodSeconds = inactivityPeriodSeconds == 0 ? 30 : Math.min(inactivityPeriodSeconds, 30);
        this.queue = new ConcurrentLinkedDeque();
        this.stats = new ConcurrentHashMap<Channel, ChannelStats>();
        if (!isMinimalPool) {
            int refreshPeriod = this.inactivityPeriodSeconds < 0 ? 30 : Math.min(30, this.inactivityPeriodSeconds);
            this.bootstrap.config().group().next().scheduleAtFixedRate((Runnable)new RefreshTask(), (long)refreshPeriod, (long)refreshPeriod, TimeUnit.SECONDS);
        }
    }

    void setKeepAlive(KeepAlive ka) {
        this.keepAlive = ka;
    }

    final Future<Channel> acquire() {
        return this.acquire(this.bootstrap.config().group().next().newPromise());
    }

    final Future<Channel> acquire(final Promise<Channel> promise) {
        block7: {
            try {
                EventLoop loop;
                Channel channel;
                while (true) {
                    if ((channel = this.queue.pollFirst()) == null) {
                        Bootstrap bs = this.bootstrap.clone();
                        ChannelFuture fut = bs.connect();
                        if (fut.isDone()) {
                            this.notifyOnConnect(fut, promise);
                        } else {
                            fut.addListener((GenericFutureListener)new ChannelFutureListener(){

                                public void operationComplete(ChannelFuture future) throws Exception {
                                    ConnectionPool.this.notifyOnConnect(future, promise);
                                }
                            });
                        }
                        return promise;
                    }
                    loop = channel.eventLoop();
                    if (!loop.inEventLoop()) break;
                    if (this.checkChannel(channel, promise)) {
                        continue;
                    }
                    break block7;
                    break;
                }
                loop.execute(new Runnable(){

                    @Override
                    public void run() {
                        ConnectionPool.this.checkChannel(channel, promise);
                    }
                });
            }
            catch (Throwable t) {
                promise.tryFailure(t);
            }
        }
        return promise;
    }

    void release(Channel channel) {
        if (!channel.isActive()) {
            LogUtil.logFine(this.logger, "Inactive channel on release, closing: " + channel);
            this.removeChannel(channel);
        }
        this.updateStats(channel, false);
        this.queue.addFirst(channel);
        try {
            this.handler.channelReleased(channel);
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void removeChannel(Channel channel) {
        this.stats.remove(channel);
        channel.close();
    }

    void close() {
        LogUtil.logFine(this.logger, "Closing pool, stats " + this.getStats());
        this.validatePool();
        Channel ch = this.queue.pollFirst();
        while (ch != null) {
            this.removeChannel(ch);
            ch = this.queue.pollFirst();
        }
        this.validatePool();
    }

    int getAcquiredChannelCount() {
        return this.acquiredChannelCount;
    }

    private void notifyOnConnect(ChannelFuture future, Promise<Channel> promise) throws Exception {
        if (future.isSuccess()) {
            Channel channel = future.channel();
            this.updateStats(channel, true);
            this.handler.channelAcquired(channel);
            if (!promise.trySuccess(channel)) {
                this.release(channel);
            }
        } else {
            promise.tryFailure(future.cause());
        }
    }

    private boolean checkChannel(Channel channel, Promise<Channel> promise) {
        if (!channel.isActive()) {
            LogUtil.logFine(this.logger, "Inactive channel found, closing: " + channel);
            this.removeChannel(channel);
            return true;
        }
        try {
            this.updateStats(channel, true);
            this.handler.channelAcquired(channel);
        }
        catch (Exception exception) {
            // empty catch block
        }
        promise.setSuccess(channel);
        return false;
    }

    int getTotalChannels() {
        return this.queue.size() + this.acquiredChannelCount;
    }

    int getFreeChannels() {
        return this.queue.size();
    }

    int pruneChannels() {
        int pruned = 0;
        long now = System.currentTimeMillis();
        for (Channel ch : this.queue) {
            if (ch.isActive()) continue;
            LogUtil.logFine(this.logger, "Channel being pruned due to server close: " + ch);
            this.queue.remove(ch);
            this.removeChannel(ch);
            ++pruned;
        }
        if (this.inactivityPeriodSeconds > 0) {
            while (this.queue.size() > this.poolMin) {
                Channel ch = this.queue.pollLast();
                ChannelStats cs = this.stats.get(ch);
                assert (cs != null);
                long inactive = (now - cs.getLastAcquired()) / 1000L;
                if (inactive > (long)this.inactivityPeriodSeconds) {
                    LogUtil.logFine(this.logger, "Channel being pruned due to inactivity: " + ch);
                    this.removeChannel(ch);
                    ++pruned;
                    continue;
                }
                this.queue.addLast(ch);
                break;
            }
        }
        this.validatePool();
        return pruned;
    }

    int doKeepAlive(int keepAlivePeriod) {
        if (this.keepAlive == null) {
            return 0;
        }
        int numToSend = this.poolMin - this.acquiredChannelCount;
        if (numToSend <= 0) {
            return 0;
        }
        long now = System.currentTimeMillis();
        int numSent = 0;
        for (Channel ch : this.queue) {
            ChannelStats cs;
            if (!ch.isActive() || (cs = this.stats.get(ch)) == null) continue;
            long inactive = (now - cs.getLastAcquired()) / 1000L;
            if (inactive >= (long)keepAlivePeriod) {
                if (!this.queue.remove(ch)) continue;
                LogUtil.logFine(this.logger, "Sending keepalive on channel " + ch + ", stats: " + cs);
                this.keepAlive.keepAlive(ch);
                cs.acquired();
                ++numSent;
                this.queue.addFirst(ch);
            }
            if (--numToSend != 0) continue;
            break;
        }
        this.validatePool();
        return numSent;
    }

    private void validatePool() {
        if (this.queue.size() + this.acquiredChannelCount != this.stats.size()) {
            LogUtil.logInfo(this.logger, "Pool count discrepancy: Queue size, acquired count, stats size :" + this.queue.size() + ", " + this.acquiredChannelCount + ", " + this.stats.size());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateStats(Channel channel, boolean isAcquire) {
        ChannelStats cstats = this.stats.get(channel);
        if (cstats == null) {
            cstats = new ChannelStats();
            this.stats.put(channel, cstats);
        }
        ConnectionPool connectionPool = this;
        synchronized (connectionPool) {
            if (isAcquire) {
                ++this.acquiredChannelCount;
                cstats.acquired();
            } else {
                --this.acquiredChannelCount;
            }
        }
    }

    void logStats() {
        LogUtil.logFine(this.logger, this.getStats());
    }

    String getStats() {
        StringBuilder sb = new StringBuilder();
        sb.append("acquiredCount=" + this.acquiredChannelCount + ", freeChannelCount=" + this.queue.size() + ", totalChannelCount=" + this.stats.size());
        sb.append(", [");
        for (Map.Entry<Channel, ChannelStats> entry : this.stats.entrySet()) {
            sb.append("channel=" + entry.getKey().id() + "[");
            entry.getValue().toStringBuilder(sb);
            sb.append("]");
        }
        sb.append("]");
        return sb.toString();
    }

    long getLastAcquired(Channel ch) {
        ChannelStats cs = this.stats.get(ch);
        if (cs != null) {
            return cs.getLastAcquired();
        }
        LogUtil.logFine(this.logger, "Can't get stats for channel " + ch.id());
        return 0L;
    }

    int getUseCount(Channel ch) {
        ChannelStats cs = this.stats.get(ch);
        if (cs != null) {
            return cs.getUseCount();
        }
        LogUtil.logFine(this.logger, "Can't get stats for channel " + ch.id());
        return 0;
    }

    private class RefreshTask
    implements Runnable {
        final int keepAlivePeriod = 30;

        private RefreshTask() {
        }

        @Override
        public final void run() {
            try {
                ConnectionPool.this.pruneChannels();
                if (ConnectionPool.this.keepAlive != null) {
                    ConnectionPool.this.doKeepAlive(30);
                }
            }
            catch (Exception e) {
                LogUtil.logFine(ConnectionPool.this.logger, "Exception in RefreshTask: " + e);
            }
        }
    }

    class ChannelStats {
        private long lastAcquired;
        private int useCount;

        ChannelStats() {
        }

        void acquired() {
            this.lastAcquired = System.currentTimeMillis();
            ++this.useCount;
        }

        long getLastAcquired() {
            return this.lastAcquired;
        }

        int getUseCount() {
            return this.useCount;
        }

        void toStringBuilder(StringBuilder sb) {
            sb.append("useCount=" + this.useCount + ", lastAcquired=" + Instant.ofEpochMilli(this.lastAcquired));
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            this.toStringBuilder(sb);
            return sb.toString();
        }
    }

    @FunctionalInterface
    static interface KeepAlive {
        public boolean keepAlive(Channel var1);
    }
}

