/*
 * Decompiled with CFR 0.152.
 */
package oracle.nosql.driver.util;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.lang.reflect.Array;
import java.math.MathContext;
import java.math.RoundingMode;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import oracle.nosql.driver.util.ByteInputStream;
import oracle.nosql.driver.util.PackedInteger;

public class SerializationUtil {
    public static final String EMPTY_STRING = new String();
    public static final byte[] EMPTY_BYTES = new byte[0];

    public static int readPackedInt(DataInput in) throws IOException {
        byte[] bytes = new byte[5];
        in.readFully(bytes, 0, 1);
        int len = PackedInteger.getReadSortedIntLength(bytes, 0);
        try {
            in.readFully(bytes, 1, len - 1);
        }
        catch (IndexOutOfBoundsException e) {
            throw new IOException("Invalid packed int", e);
        }
        return PackedInteger.readSortedInt(bytes, 0);
    }

    public static int skipPackedInt(ByteInputStream in) throws IOException {
        byte b = in.readByte();
        int len = PackedInteger.getReadSortedIntLength(new byte[]{b}, 0);
        if (len > 1) {
            in.skip(len - 1);
        }
        return len;
    }

    public static int writePackedInt(DataOutput out, int value) throws IOException {
        byte[] buf = new byte[5];
        int offset = PackedInteger.writeSortedInt(buf, 0, value);
        out.write(buf, 0, offset);
        return offset;
    }

    public static long readPackedLong(ByteInputStream in) throws IOException {
        byte[] bytes = new byte[9];
        in.readFully(bytes, 0, 1);
        int len = PackedInteger.getReadSortedLongLength(bytes, 0);
        try {
            in.readFully(bytes, 1, len - 1);
        }
        catch (IndexOutOfBoundsException e) {
            throw new IOException("Invalid packed long", e);
        }
        return PackedInteger.readSortedLong(bytes, 0);
    }

    public static int skipPackedLong(ByteInputStream in) throws IOException {
        byte b = in.readByte();
        int len = PackedInteger.getReadSortedLongLength(new byte[]{b}, 0);
        if (len > 1) {
            in.skip(len - 1);
        }
        return len;
    }

    public static void writePackedLong(DataOutput out, long value) throws IOException {
        byte[] buf = new byte[9];
        int offset = PackedInteger.writeSortedLong(buf, 0, value);
        out.write(buf, 0, offset);
    }

    public static String readString(ByteInputStream in) throws IOException {
        return SerializationUtil.readStdUTF8String(in);
    }

    public static int skipString(ByteInputStream in) throws IOException {
        int start = in.getOffset();
        int len = SerializationUtil.readPackedInt(in);
        if (len > 0) {
            in.skip(len);
        }
        return in.getOffset() - start;
    }

    public static String readNonNullString(ByteInputStream in) throws IOException {
        String result = SerializationUtil.readString(in);
        if (result == null) {
            throw new IOException("Found null value for non-null string");
        }
        return result;
    }

    private static String readStdUTF8String(ByteInputStream in) throws IOException {
        int length = SerializationUtil.readPackedInt(in);
        if (length < -1) {
            throw new IOException("Invalid length of String: " + length);
        }
        if (length == -1) {
            return null;
        }
        if (length == 0) {
            return EMPTY_STRING;
        }
        byte[] bytes = new byte[length];
        in.readFully(bytes);
        return StandardCharsets.UTF_8.decode(ByteBuffer.wrap(bytes)).toString();
    }

    public static int writeString(DataOutput out, String value) throws IOException {
        return SerializationUtil.writeStdUTF8String(out, value);
    }

    public static void writeNonNullString(DataOutput out, String value) throws IOException {
        SerializationUtil.checkNull("value", value);
        SerializationUtil.writeString(out, value);
    }

    private static int writeStdUTF8String(DataOutput out, String value) throws IOException {
        if (value == null) {
            return SerializationUtil.writePackedInt(out, -1);
        }
        ByteBuffer buffer = StandardCharsets.UTF_8.encode(value);
        int length = buffer.limit();
        int len = SerializationUtil.writePackedInt(out, length);
        if (length > 0) {
            out.write(buffer.array(), 0, length);
        }
        return len + length;
    }

    public static int readSequenceLength(ByteInputStream in) throws IOException {
        int result = SerializationUtil.readPackedInt(in);
        if (result < -1) {
            throw new IOException("Invalid sequence length: " + result);
        }
        return result;
    }

    public static int readNonNullSequenceLength(ByteInputStream in) throws IOException {
        int length = SerializationUtil.readSequenceLength(in);
        if (length == -1) {
            throw new IOException("Read null length for non-null sequence");
        }
        return length;
    }

    public static void writeSequenceLength(DataOutput out, int length) throws IOException {
        if (length < -1) {
            throw new IllegalArgumentException("Invalid sequence length: " + length);
        }
        SerializationUtil.writePackedInt(out, length);
    }

    public static void writeNonNullSequenceLength(DataOutput out, int length) throws IOException {
        if (length < 0) {
            throw new IllegalArgumentException("Invalid non-null sequence length: " + length);
        }
        SerializationUtil.writePackedInt(out, length);
    }

    public static byte[] readByteArray(ByteInputStream in) throws IOException {
        int len = SerializationUtil.readSequenceLength(in);
        if (len < -1) {
            throw new IOException("Invalid length of byte array: " + len);
        }
        if (len == -1) {
            return null;
        }
        if (len == 0) {
            return EMPTY_BYTES;
        }
        byte[] array = new byte[len];
        in.readFully(array);
        return array;
    }

    public static int skipByteArray(ByteInputStream in) throws IOException {
        int start = in.getOffset();
        int len = SerializationUtil.readSequenceLength(in);
        if (len > 0) {
            in.skip(len);
        }
        return in.getOffset() - start;
    }

    public static void writeByteArray(DataOutput out, byte[] array) throws IOException {
        int length = array == null ? -1 : Array.getLength(array);
        SerializationUtil.writeSequenceLength(out, length);
        if (length > 0) {
            out.write(array);
        }
    }

    public static byte[] readNonNullByteArray(ByteInputStream in) throws IOException {
        byte[] array = SerializationUtil.readByteArray(in);
        if (array == null) {
            throw new IOException("Read unexpected null array");
        }
        return array;
    }

    public static void writeNonNullByteArray(DataOutput out, byte[] array) throws IOException {
        SerializationUtil.checkNull("array", array);
        SerializationUtil.writeByteArray(out, array);
    }

    public static void writePackedIntArray(DataOutput out, int[] array) throws IOException {
        int len = array == null ? -1 : array.length;
        SerializationUtil.writeSequenceLength(out, len);
        if (array != null) {
            for (int v : array) {
                SerializationUtil.writePackedInt(out, v);
            }
        }
    }

    public static int[] readPackedIntArray(ByteInputStream in) throws IOException {
        int len = SerializationUtil.readSequenceLength(in);
        if (len < -1) {
            throw new IOException("Invalid length of byte array: " + len);
        }
        if (len == -1) {
            return null;
        }
        int[] array = new int[len];
        for (int i = 0; i < len; ++i) {
            array[i] = SerializationUtil.readPackedInt(in);
        }
        return array;
    }

    public static void writeIntArray(DataOutput out, int[] array) throws IOException {
        int len = array == null ? -1 : array.length;
        SerializationUtil.writeSequenceLength(out, len);
        if (array != null) {
            for (int v : array) {
                out.writeInt(v);
            }
        }
    }

    public static int[] readIntArray(ByteInputStream in) throws IOException {
        int len = SerializationUtil.readSequenceLength(in);
        if (len < -1) {
            throw new IOException("Invalid length of byte array: " + len);
        }
        if (len == -1) {
            return null;
        }
        int[] array = new int[len];
        for (int i = 0; i < len; ++i) {
            array[i] = in.readInt();
        }
        return array;
    }

    public static String[] readStringArray(ByteInputStream in) throws IOException {
        int len = SerializationUtil.readSequenceLength(in);
        if (len < -1) {
            throw new IOException("Invalid length of byte array: " + len);
        }
        if (len == -1) {
            return null;
        }
        String[] array = new String[len];
        for (int i = 0; i < len; ++i) {
            array[i] = SerializationUtil.readString(in);
        }
        return array;
    }

    public static void writeMathContext(MathContext mathContext, DataOutput out) throws IOException {
        if (mathContext == null) {
            out.writeByte(0);
        } else if (MathContext.DECIMAL32.equals(mathContext)) {
            out.writeByte(1);
        } else if (MathContext.DECIMAL64.equals(mathContext)) {
            out.writeByte(2);
        } else if (MathContext.DECIMAL128.equals(mathContext)) {
            out.writeByte(3);
        } else if (MathContext.UNLIMITED.equals(mathContext)) {
            out.writeByte(4);
        } else {
            out.writeByte(5);
            out.writeInt(mathContext.getPrecision());
            out.writeInt(mathContext.getRoundingMode().ordinal());
        }
    }

    public static MathContext readMathContext(DataInput in) throws IOException {
        byte code = in.readByte();
        switch (code) {
            case 0: {
                return null;
            }
            case 1: {
                return MathContext.DECIMAL32;
            }
            case 2: {
                return MathContext.DECIMAL64;
            }
            case 3: {
                return MathContext.DECIMAL128;
            }
            case 4: {
                return MathContext.UNLIMITED;
            }
            case 5: {
                int precision = in.readInt();
                int roundingMode = in.readInt();
                return new MathContext(precision, RoundingMode.valueOf(roundingMode));
            }
        }
        throw new IOException("Unknown MathContext code.");
    }

    private static void checkNull(String variableName, Object value) {
        if (value == null) {
            throw new IllegalArgumentException("The value of " + variableName + " must not be null");
        }
    }
}

