package io.flutter.add2apptest;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.util.Log;

import androidx.annotation.NonNull;

import io.flutter.embedding.engine.FlutterEngine;
import io.flutter.embedding.engine.FlutterEngineCache;
import io.flutter.embedding.engine.dart.DartExecutor;
import io.flutter.plugin.common.JSONMethodCodec;
import io.flutter.plugin.common.MethodChannel;

import io.flutter.view.FlutterMain;

public final class FlutterDelegate {
    // Per-class log tags make logs more legible
    @NonNull
    public static final String LOG_TAG = FlutterDelegate.class.getSimpleName();

    private static final String ENGINE_ID = "TestEngine";

    private boolean isInitialized;
    private boolean isClosing;

    private static FlutterDelegate instance;

    private MethodChannel controlChannel;

    private FlutterEngine flutterEngine;
    private MyFlutterActivity flutterActivity;
    private String launchingRoute;
    private Activity launchingActivity;

    private void setUpFlutterEngineWithEntryPoint(@NonNull Context context) {
        flutterEngine = new FlutterEngine(context);

        flutterEngine.getDartExecutor().executeDartEntrypoint(DartExecutor.DartEntrypoint.createDefault());
        FlutterEngineCache.getInstance().put(ENGINE_ID, flutterEngine);

        // Setup control channel to synchronize
        controlChannel = new MethodChannel(flutterEngine.getDartExecutor(), "io.flutter.add2apptest.control", JSONMethodCodec.INSTANCE);
        controlChannel.setMethodCallHandler((call, result) -> {
            switch (call.method) {
                case "initialize":
                    result.success(null);
                    controlChannel.invokeMethod("initialized", null, new MethodChannel.Result() {
                        @Override
                        public void success(Object obj) {
                            onInitialized();
                        }

                        @Override
                        public void error(String code, String msg, Object obj) {
                            Log.e(LOG_TAG, "sending initialized message failed!");
                        }

                        @Override
                        public void notImplemented() {
                            Log.e(LOG_TAG, "sending initialized message failed because method is not implemented on Flutter side!");
                        }
                    });
                    break;
                case "prepareForPop":
                    if (flutterActivity != null) {
                        isClosing = true;
                        result.success(null);
                    }
                    else {
                        result.error("no_flutter_activity", "No FlutterActivity present", null);
                    }
                    break;
                case "pop":
                    if (flutterActivity != null) {
                        flutterActivity.finish();
                        result.success(null);
                    }
                    else {
                        result.error("no_flutter_activity", "No FlutterActivity present", null);
                    }
                    break;
            }
        });
    }

    public static @NonNull FlutterDelegate getInstance(Context context) {
        if (instance == null && context != null) {
            instance = new FlutterDelegate();
            instance.initFlutterEngine(context);
        }
        return instance;
    }

    String getFlutterEngineId() {
        return ENGINE_ID;
    }

    public FlutterEngine getFlutterEngine() {
        return flutterEngine;
    }

    // This is called by the FlutterActivity been started, so FlutterDelegate can manage its life
    // time. Only one FlutterActivity can exist.
    void setFlutterActivity(MyFlutterActivity activity) {
        flutterActivity = activity;
        if (flutterActivity != null) {
            launchingRoute = null;
            launchingActivity = null;
        }
        else {
            isClosing = false;
            if (launchingRoute != null) {
                launchFlutterActivity();
            }
        }
    }

    private void initFlutterEngine(Context context) {
        FlutterMain.startInitialization(context);

        FlutterMain.ensureInitializationComplete(context, null);

        setUpFlutterEngineWithEntryPoint(context);
    }

    // FlutterEngine has been initialed and the platform style resources have been exported to Flutter side.
    private void onInitialized() {
        isInitialized = true;
        if (flutterActivity == null && launchingRoute != null) {
            launchFlutterActivity();
        }
    }

    // Pushes the flutter route described by the route parameter
    void pushFlutterRoute(@NonNull String route, Activity launchingActivity) {
        if (flutterActivity == null || isClosing) {
            launchingRoute = route;
            this.launchingActivity = launchingActivity;
            if (isInitialized && flutterActivity == null) {
                launchFlutterActivity();
            }
        }
        else {
            flutterEngine.getNavigationChannel().pushRoute(route);
        }
    }

    public void popFlutterRoute() {
        if (flutterActivity != null) {
            flutterEngine.getNavigationChannel().popRoute();
        }
    }

    private void launchFlutterActivity() {
        if (flutterActivity == null) {
            // push the new launching route
            flutterEngine.getNavigationChannel().pushRoute(launchingRoute);

            Intent intent = new Intent(launchingActivity, MyFlutterActivity.class);
            launchingActivity.startActivity(intent);
        }
    }

    private void destroyFlutterEngine() {
        isInitialized = false;

        controlChannel.setMethodCallHandler(null);
        controlChannel = null;

        if (flutterActivity != null) {
            flutterActivity.finish();
            flutterActivity = null;
        }

        if (flutterEngine != null) {
            flutterEngine.destroy();
            flutterEngine = null;
            FlutterEngineCache.getInstance().remove(ENGINE_ID);
        }
    }
}
